#ifndef __TRandomNumberGenerator__
#define __TRandomNumberGenerator__

#include "CMathTools.hpp"
#include "../Collections/TCollection.hpp"
#include <cstdlib>
#include <ctime>
using Exponent::Collections::TCollection;
//using Exponent::MathTools::CRandomNumberGenerator;
using Exponent::MathTools::CMathTools;

//	===========================================================================

namespace Exponent
{
	namespace MathTools
	{
		/**
		 * @class TRandomNumberGenerator TRandomNumberGenerator.hpp
		 * @brief Generates a (not too) random floating point number
		 *
		 * @date 02/11/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk\n\n
		 * \n
		 * The Gaussian1 and Uniform code comes from the book C++ algorithms for DSP (2nd ED.)\n
		 * By Ambree and Danieli. It generates a gaussian (white) noise source using the \n
		 * Box-Muller method (see Knuth or Press et al)\n
		 * ISBN : 0-13-179144-3\n\n
		 * The Guassian2 function is based upon tobybears original noise generator, i just\n
		 * converted it to C++ from the original delphi.\n
		 * Link for original is: http://www.musicdsp.org/archive.php?classid=0#129 \n\n
		 * The Gaussian3 function is based upon the work of Steve S Smith in\n
		 * The Scientist and Engineer's Guide to Digital Signal Processing\n
		 * which is available at http://www.dspguide.com \n
		 *
		 * $Id: TRandomNumberGenerator.hpp,v 1.4 2007/02/08 21:06:44 paul Exp $
		 */
		template <class TypeName> class TRandomNumberGenerator : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * @enum ERandomNumberGenerationType
			 * @brief Type of random number
			 */
			enum ERandomNumberGenerationType
			{
				e_standardLibrary = 0,				/**< The standard library type of random number, not very random */
				e_uniform,							/**< Uniform distribution */
				e_gaussian1,						/**< Gaussian distribution variant 1 */
				e_gaussian2,						/**< Gaussian distribution variant 2 */
				e_gaussian3							/**< Gaussian distribution variant 3 */
			};

//	===========================================================================

			/**
			 * Construction
			 */
			TRandomNumberGenerator()
			{
				EXPONENT_CLASS_CONSTRUCTION(TRandomNumberGenerator<TypeName>);
			}

			/**
			 * Destruction
			 */
			virtual ~TRandomNumberGenerator()
			{
				EXPONENT_CLASS_DESTRUCTION(TRandomNumberGenerator<TypeName>);
			}

//	===========================================================================

			/**
			 * Seed the signal
			 * @note Call this once before starting to generate numbers
			 */
			void seed()
			{
				srand((unsigned int)time(NULL));
			}

//	===========================================================================

			/**
			 * Get a new random number
			 * @param type The type of random number
			 * @retval TypeName The new random number in range 0 - 1
			 */
			TypeName getNewRandomNumber(const ERandomNumberGenerationType type)
			{
				switch(type)
				{
					case e_standardLibrary:		return this->standardLibrary();		break;
					case e_uniform:				return this->uniform();				break;
					case e_gaussian1:			return this->gaussian1();			break;
					case e_gaussian2:			return this->gaussian2();			break;
					case e_gaussian3:			return this->gaussian3();			break;
				}
				return 0;
			}

			/**
			 * Fill an array with random numbers (Fills with doubles up to size)
			 * @param array The array to fill with CDoubles
			 * @param numberToGenerate Total number of random numbers to generate. Should be the number of elements in array
			 * @param type The type of random numbers
			 */
			void getNewRandomArray(TypeName *array, const long numberToGenerate, const ERandomNumberGenerationType type)
			{
				// Check size
				if (array && numberToGenerate >= 1)
				{
					// Handle based on type
					switch(type)
					{
						case e_standardLibrary:
							for (long i = 0; i < numberToGenerate; i++)
							{
								array[i] = this->standardLibrary();
							}
						break;
						case e_uniform:
							for (long i = 0; i <numberToGenerate; i++)
							{
								array[i] = this->uniform();
							}
						break;
						case e_gaussian1:
							for (long i = 0; i < numberToGenerate; i++)
							{
								array[i] = this->gaussian1();
							}
						break;
						case e_gaussian2:
							for (long i = 0; i < numberToGenerate; i++)
							{
								array[i] = this->gaussian2();
							}
						break;
						case e_gaussian3:
							for (long i = 0; i < numberToGenerate; i++)
							{
								array[i] = this->gaussian3();
							}
						break;
					}
				}
			}

			/**
			 * Fill an array with random numbers (Fills with doubles up to size)
			 * @param array The array to fill with elements of TypeName. Note that this array must be initialised to the correct size
			 * @param type The type of random numbers
			 */
			void getNewRandomArray(TCollection<TypeName> &array, const ERandomNumberGenerationType type)
			{
				this->getNewRandomArray(array.getMutableInternalBuffer(), array.getArraySize(), type);
			}

//	===========================================================================

		protected:

//	===========================================================================

			const static TypeName TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[3];						/**< The variables required for gaussian 2 */

//	===========================================================================

			/**
			 * Generate a random number between -1 and 1
			 * @retval TypeName The random number
			 */
			TypeName uniform()
			{
				return (TypeName)(((TypeName)(rand() & RAND_MAX) / RAND_MAX - 0.5) * 2.0);
			}

			/**
			 * Generate a std lib random number between 0 and 1
			 * @retval TypeName The random number
			 */
			TypeName standardLibrary()
			{
				return ((TypeName)rand()) / (TypeName)RAND_MAX;
			}

			/**
			 * Generate gaussian random number
			 * @retval TypeName The gaussian number
			 */
			TypeName gaussian1()
			{
				// Flag to indicate if state is stored already
				static bool ready = false;

				// Variables required to store gaus value
				static double gaussianStore;

				// Other variables
				TypeName vertex1;
				TypeName vertex2;
				TypeName radius;
				TypeName fractional;
				TypeName gaussian;

				if (!ready)
				{
					do
					{
						vertex1 = uniform();
						vertex2 = uniform();
						radius = CMathTools::square(vertex1) + CMathTools::square(vertex2);
					}
					while(radius > (TypeName)1.0);

					// Compute two gaussian numbers
					fractional = CMathTools::fastSquareRoot(1.0 * -CMathTools::fastLog(radius) / radius);

					// Store one
					gaussianStore = vertex1 * fractional;

					// Store the other
					gaussian = vertex2 * fractional;

					// We dont need to generate next time
					ready = true;
				}
				else
				{
					ready    = false;
					gaussian = gaussianStore;
				}

				return gaussian;
			}

			/**
			 * Generate gaussian random number
			 * @retval TypeName The gaussian number
			 */
			TypeName gaussian2()
			{
				const TypeName random           = ((TypeName)rand() / (TypeName)(RAND_MAX + 1));
				const TypeName randomMultiplied = random * TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[1];
				return (TypeName)((2.0 * (randomMultiplied + randomMultiplied + randomMultiplied) - 3.0 * (TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[1] - 1.0)) * TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[2]);
			}

			/**
			 * Generate gaussian random number
			 * @retval TypeName The gaussian number
			 */
			TypeName gaussian3()
			{
				return CMathTools::fastSquareRoot(-2.0 * CMathTools::fastLog(this->standardLibrary())) * cos(CMathTools::CMATH_2PI_DOUBLE * this->standardLibrary());
			}
		};

		/** @cond */
		EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TRandomNumberGenerator<TypeName>, TypeName, CCountedObject);
		template<class TypeName>const TypeName TRandomNumberGenerator<TypeName>::TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[] =
		{
			(1 << 15) - 1,
			((int)(TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[0] / 3)) + 1,
			1.0 / TRANDOM_NUMBER_GENERATOR_GAUSSIAN_COEFFICIENTS[1]
		};
		/** @endcond */
	}
}
#endif	// End of TRandomNumberGenerator.hpp